
package cz.luboss.medulla.common;

import java.util.Locale;
import java.util.ResourceBundle;

// definice
import cz.luboss.medulla.data.*;
import cz.luboss.medulla.factory.MeDataMachine;

/**
 *  Pracovn kontext pro Medulla
 *
 *@author    LKC
 */
public class MeContext {

	/** Konfigurace s paramerty pripojeni */
	protected MeConfig m_Config;
  /** Masina pro praci s daty */
	protected MeDataMachine m_DataMachine;
  /** Uzivatelska session */
	protected MeSession m_Session;
	/** Soubor s texty v jazykovych variantach */
	protected ResourceBundle m_rResources = null;
	/** Transakcni kontext */
	protected MeTransact m_tTransact = MeTransact.CreateInstance();

	/**
	 *  Nastavi konfiguraci
	 *
	 *@param  iConfig  konfigurace
	 *@return          prubeh funkce
	 */
	public boolean SetConf(MeConfig iConfig) {
		this.m_Config = iConfig;
		return true;
	}

	/**
	 *  Nastavi masinu pro praci s daty
	 *
	 *@param  iDataMachine  masina pro praci s daty
	 *@return               prubeh funkce
	 */
	public boolean SetMachine(MeDataMachine iDataMachine) {
		this.m_DataMachine = iDataMachine;
		return true;
	}

	/**
	 *  Nastav uzivatelskou session
	 *
	 *@param  iSession  uzivatelska session
	 *@return           prubeh funkce
	 */
	public boolean SetSession(MeSession iSession) {
		this.m_Session = iSession;
		// a zaroven prenastavi resources podle uzivatelova jazyka
    // TODO: vyhazet odkazy na pensum
		switch (m_Session.getLang()) {
			case MeDefine.DEF_LANG_CZECH:
				m_rResources = ResourceBundle.getBundle("cz.luboss.pensum.common.PeResources", new Locale("cs"));
				break;
			case MeDefine.DEF_LANG_ENGLISH:
				m_rResources = ResourceBundle.getBundle("cz.luboss.pensum.common.PeResources", Locale.ENGLISH);
				break;
			case MeDefine.DEF_LANG_GERMAN:
				m_rResources = ResourceBundle.getBundle("cz.luboss.pensum.common.PeResources", Locale.GERMAN);
				break;
		}

		return true;
	}


	/**
	 *  Vrati nastaveni konfigurace
	 *
	 *@return    viz popis
	 */
	public MeConfig GetConf() {
		return m_Config;
	}

	/**
	 *  Vrati masinu pro praci s daty
	 *
	 *@return    viz popis
	 */
	public MeDataMachine GetMachine() {
		return m_DataMachine;
	}

	/**
	 *  Vrati uzivatelskou session
	 *
	 *@return    viz popis
	 */
	public MeSession GetSession() {
		return m_Session;
	}

	/**
	 *  Vrati text popisu podle jazyka uzivatele
	 *
	 *@param  sName  jmeno textu
	 *@return        viz popis
	 */
	public String GetRes(String sName) {
		if (GetSession() == null) {
			return "Unknow user";
		}
		return m_rResources.getString(sName);
	}

	/**
	 *  Vytvori novy objet
	 *
	 *@return    viz popis
	*/
	public static MeContext CreateInstance() {
		MeContext context = new MeContext();
		return context;
	}

  /**
	 * Zacne novou transakci
	 *
	 * @return prubeh funkce
	*/
	public boolean StartTransaction() {
    return m_tTransact.StartTransaction();
  }
  
	/**
	 * Vrati vsechny prvky zmenene v teto transakci do puvodniho stavu
	 *
	 * @return vrati prubeh fukce
	*/
	public boolean RollBack() {
    return m_tTransact.RollBack();
  }

	/**
	 * Potvrdi vsechny provedene zmeny
	 *
	 * @return vrati prubeh fukce
	*/
	public boolean Commit() {
    // pokud je to posledni transakce, ulozi celou transakci 
    if (m_tTransact.TransactionIsLast()
      && !this.GetMachine().SaveTrans(this.GetSession().getID(), m_tTransact.GetTransactionData())) {
      return false;
    }
    return m_tTransact.Commit();
  }

	/**
	 * Zaregistruje objekt do transakce
	 *
	 * @return vrati prubeh fukce
	*/
	public boolean Register(MeItem item) {
    // pokud nema ID, dotahneme z databaze
    if (!item.isIDGenerate()) {
      item.addSequenceID();
    }
    return m_tTransact.Register(item);
  }

	/**
	 * Odregistruje objekt z transakce
	 *
	 * @return vrati prubeh fukce
	*/
	public boolean UnRegister(MeItem item) {
    return m_tTransact.UnRegister(item);
  }

  /**
	 * Nahrajeme objekt
	 *
   * @param item objekt
   * @param nID ID objekt
	 * @return viz popis
	*/
  public boolean LoadData(MeItem item, int nID) {
    // pokud jejiz dotazem
    if (m_tTransact.IsInActual(item.generateKod(nID))) {
      item.setData(m_tTransact.GetFromActual(item.generateKod(nID)));
    } else {
      // jinak dotahneme
      if (!this.GetMachine().LoadData(this.GetSession().getID(), item, nID)) {
        return false;
      }
      // ulozime do transakce
      m_tTransact.SetToActual(item);
      // nastavime status
  		item.setStatus(MeDefine.DEF_STATUS_LOAD);
    }
    return true;
	}

  /**
	 * Nahrajeme pole objektu
	 *
   * @param array pole
   * @param iPodminka podlminka vyberu
	 * @return viz popis
	*/
  public boolean LoadArray(MeArray array, MePodminka iPodminka) {
    if (!MedullaContext.GetMachine().LoadArray(MedullaContext.GetSessionID(), array, iPodminka)) {
      return false;
    }
    // ted je potreba vsem prvkum nastvit hodnotu na nahrano
    array.setStatus(MeDefine.DEF_STATUS_LOAD);
    // TODO: a tady bych jednou mel dodelat prace s transakci
    // 1 : dotahnout aktualni hodnoty objektu v poli z transakce
    // 2 : pridat/vyjmout objekty zmenene v transakci, ktere jiz do podminky patri
    return true;
  }
  
  /**
	 * Smaze objekt
	 *
   * @param item objekt
   * @param nID ID objekt
	 * @return viz popis
	*/
  public boolean DeleteData(MeItem item, int nID) {
		item.setStatus(MeDefine.DEF_STATUS_DELETED);
    return true;
	}

}
